// *************************************************************************
// freiesMagazin-Programmierwettbewerb (ai)
// Copyright 2009 Dominik Wagenfuehr <dominik.wagenfuehr@deesaster.org>
// Licence: GPLv3
// *************************************************************************

/**
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// System
///////////
#include <iostream>
#include <fstream>
#include <vector>

// Own
//////////////
#include "gamefield.h"
#include "fieldpos.h"

// standard constructor
// initialize game field
GameField::GameField(void)
{
    // nothing to do
}

// Copy constructor
GameField::GameField(const GameField& field)
{
    for ( unsigned int i = 0; i < FIELD_WIDTH; i++ )
    {
        for ( unsigned int j = 0; j < FIELD_HEIGHT; j++ )
        {
            m_field[i][j] = field.m_field[i][j];
        }
    }
}
      
// destructor
GameField::~GameField(void)
{
    // nothing to do
}

// copy game field
void GameField::set(const GameField& field)
{
    for ( unsigned int i = 0; i < FIELD_WIDTH; i++ )
    {
        for ( unsigned int j = 0; j < FIELD_HEIGHT; j++ )
        {
            m_field[i][j] = field.m_field[i][j];
        }
    }  
}

// opens file and reads the first 100 relevant characters
// newline, cr, spaces and tabs will be ignored
// return true if everything is ok
const bool GameField::read(const std::string& filename)
{
    bool ok = false;

    if ( !filename.empty() )
    {
        // open file for writing
        std::ifstream infile( filename.c_str() );
        
        if ( NULL != infile )
        {
            unsigned int i = 0;
            unsigned int j = 0;
            
            while ( infile.good() && j < FIELD_HEIGHT )
            {
                if ( i >= FIELD_WIDTH )
                {
                    i = 0;
                    j++;
                }
                
                // read character
                char c = 0;
                infile.get(c);
                // std::cout << c;
                
                switch ( (Tile::TileType)c )
                {
                    case Tile::TILE_RED:
                    case Tile::TILE_GREEN:
                    case Tile::TILE_YELLOW:
                    case Tile::TILE_SHIELD:
                    case Tile::TILE_LILAC:
                    case Tile::TILE_BOMB_1:
                    case Tile::TILE_BOMB_2:
                    case Tile::TILE_BOMB_3:
                    case Tile::TILE_BOMB_4:
                    case Tile::TILE_BOMB_5:
                        // store tile
                        m_field[i][FIELD_HEIGHT-j-1].set( (Tile::TileType)c );
                        i++;
                        break;
                    default:
                      // ignore everything else
                      break;
                }
            }
            
            // check if whole field was loaded
            if ( ( 0 != i ) || ( FIELD_HEIGHT != j ) )
            {
                std::cout << "GameField::read() error: game field was not loaded completely" << std::endl;
            }
            else
            {
                ok = true;
            }

            // close file
            infile.close();
        }
        else
        {
            std::cout << "GameField::read() error: file could not be opened" << std::endl;
        }
    }
    else
    {
        std::cout << "GameField::read() error: string is empty" << std::endl;
    }
    
    if (ok)
    {
        // print();
    }

    return ok;
}

// prints the game field on screen
void GameField::print(void) const
{
    std::cout << "  | 0 1 2 3 4 5 6 7 8 9" << std::endl;
    std::cout << "--+--------------------" << std::endl;

    for ( unsigned int j = 0; j < FIELD_HEIGHT; j++ )
    {
        std::cout << FIELD_HEIGHT-j-1 << " | ";

        for ( unsigned int i = 0; i < FIELD_WIDTH; i++ )
        {
            // initialize with zeros
            std::cout << (char)m_field[i][FIELD_HEIGHT-j-1].getType() << " ";
        }
        std::cout << std::endl;
    }        
    std::cout << std::endl;
}

// search all tiles on game field that could be swapped
// return true if resulting array is not empty
const bool GameField::findSwapPositions(SwappingPosArray& positions) const
{
    // we go through each column and row and swap single elements
    for ( unsigned int i = 0; i < FIELD_WIDTH-1; i++ )
    {
        for ( unsigned int j = 0; j < FIELD_HEIGHT-1; j++ )
        {
            // set positions for swapping in columns
            SwappingPos pos1( FieldPos(i,j), FieldPos(i,j+1) );
            
            // swap tiles at positions and check if there
            // are same tiles connected
            if ( checkSwapPosition(pos1) )
            {
                // store swap position
                positions.push_back(pos1);
            }
            
            // set positions for swapping in rows
            SwappingPos pos2( FieldPos(i,j), FieldPos(i+1,j) );
            
            // swap tiles at positions and check if there
            // are same tiles connected now
            if ( checkSwapPosition(pos2) )
            {
                positions.push_back(pos2);
            }            
        }
    }
    
    bool ok = false;
    
    // check if array is empty
    if ( !positions.empty() )
    {
        ok = true;
    }
    
    return ok;
}

// swap tiles at positions and check if there are same tiles connected
// return true if position could be found
const bool GameField::checkSwapPosition(SwappingPos& pos) const
{
    bool found = false;

    // copy game field
    GameField tempField(*this); 

    // swap tiles
    if ( tempField.swapTiles( pos ) )
    {
        // check if there are some same tiles
        // if ( tempField.findSameTiles() )
        if ( tempField.findSameTiles( pos ) )
        {
            found = true;
        }
    }

    return found;
}

// return const reference for tile    
const Tile& GameField::getTile(const FieldPos& pos) const
{
    if ( ( pos.x() >= FIELD_WIDTH ) || ( pos.y() >= FIELD_HEIGHT ) )
    {
        std::cout << "GameField::getTile() access out of boundaries" << std::endl;
        return m_dummyTile;
    }
    else
    {
        return m_field[pos.x()][pos.y()];
    }
}

// return reference for tile    
Tile& GameField::getTile(const FieldPos& pos)
{
    if ( ( pos.x() >= FIELD_WIDTH ) || ( pos.y() >= FIELD_HEIGHT ) )
    {
        std::cout << "GameField::getTile() access out of boundaries" << std::endl;
        return m_dummyTile;
    }
    else
    {
        return m_field[pos.x()][pos.y()];
    }
}

// tries to swap two tiles
// return true if possible
const bool GameField::swapTiles(const SwappingPos& pos)
{
    bool possible = false;
    
    FieldPos pos1(pos.first());
    FieldPos pos2(pos.second());
    
    // check if two fields are inside array boundaries
    if ( pos1.x() < FIELD_WIDTH && pos1.y() < FIELD_HEIGHT &&
         pos2.x() < FIELD_WIDTH && pos2.y() < FIELD_HEIGHT )
    {
        // check if next to each other
        if ( ( (pos1.x() == pos2.x()) && ((pos1.y() == pos2.y()+1 || pos1.y()+1 == pos2.y())) ) ||
             ( (pos1.y() == pos2.y()) && ((pos1.x() == pos2.x()+1 || pos1.x()+1 == pos2.x())) ) )
        {
            // swap tiles
            getTile(pos1).swap( getTile(pos2) );
            possible = true;
        }
        else
        {
            std::cout << "GameField::swapTiles() error: tiles not next to each other"
                      << std::endl;
        }
    }
    else
    {
        std::cout << "GameField::swapTiles() error: pos out of boundaries"
                  << std::endl;
    }

    return possible;
}

// search for equal tiles at special position
// return true if something has been found
const bool GameField::findSameTiles( const SwappingPos& pos, ScoredTileArray* array ) const
{
    bool found = false;

    if ( pos.isSwapInRow() )
    {
        // the both positions swap two tiles in a row
        // so we only need to check this row and both columns
        found = findSameTilesInRow( pos.first().y(), array );
        found = findSameTilesInColumn( pos.first().x(), array ) | found;
        found = findSameTilesInColumn( pos.second().x(), array ) | found;
    }
    else if ( pos.isSwapInColumn() )
    {
        // the both positions swap two tiles in a column
        // so we only need to check this columns and both rows
        found = findSameTilesInRow( pos.first().y(), array );
        found = findSameTilesInRow( pos.second().y(), array ) | found;
        found = findSameTilesInColumn( pos.first().x(), array ) | found;
    }
    else
    {
        // the two tiles are not connected to each other
        // this should not happen
        std::cout << "GameField::findSameTiles() error: positions are not connected"
                  << std::endl;
  
        // as fallback mode we use the global algorithm
        // that checks each field
        found = findSameTiles( array );
    }
    
    return found;
}


// search for equal tiles
// return true if something has been found
const bool GameField::findSameTiles( ScoredTileArray* tArray ) const
{
    bool found = false;

    if ( findSameTilesInColumns(tArray) )
    {
        found = true;
    }
    
    if ( findSameTilesInRows(tArray) )
    {
        found = true;
    }

    return found;
}   

// search for equal tiles in rows
// return true if something has been found
const bool GameField::findSameTilesInRows(ScoredTileArray* tArray) const
{
    bool found = false;
      
    // now go through each row
    for ( unsigned int j = 0; j < FIELD_HEIGHT; j++ )
    {
        found = findSameTilesInRow( j, tArray ) | found;

        // if we have found a position and no array is defined
        // we do not need to search further       
        if ( found && (NULL == tArray) )
        {
            break;
        }
        
    } // for FIELD_HEIGHT     
    
    return found;
}   

// search for equal tiles in columns
// return true if something has been found
const bool GameField::findSameTilesInColumns(ScoredTileArray* tArray) const
{
    bool found = false;
            
    // we will go through each column
    for ( unsigned int i = 0; i < FIELD_WIDTH; i++ )
    {
        found = findSameTilesInColumn( i, tArray ) | found;

        // if we have found a position and no array is defined
        // we do not need to search further       
        if ( found && (NULL == tArray) )
        {
            break;
        }
    }    
    
    return found;
}   

// search for equal tiles in single row
// return true if something has been found
const bool GameField::findSameTilesInRow( const unsigned int pos, ScoredTileArray* tArray ) const
{
    // because this is a internal function we suppose, that pos is
    // between 0 and FIELD_HEIGHT

    bool found = false;
      
    // reset length
    unsigned int length = 0;
    unsigned int bombValue = 0;

    for ( unsigned int i = 0; i < FIELD_WIDTH; i++ )
    {
        if ( ( !m_field[i][pos].isEmpty() ) &&
             ( 0 == i || m_field[i][pos] == m_field[i-1][pos] ) )
        {
            // same tile as before or new start tile
            length++;
            
            // check if bomb and add bomb value
            if ( m_field[i][pos].isBomb() )
            {
                bombValue += m_field[i][pos].getBombValue();
            }
            
            // if the last tile in this column
            // we must check the length
            if ( FIELD_WIDTH-1 == i && length >= 3 )
            {
                found = true;

                // std::cout << "GameField::findSameTilesInRows()    "
                //           << i-length+1 << " " 
                //           << pos << " "
                //           << length << " "
                //           << (char)m_field[i][j].getType() << " "
                //           << bombValue
                //           << std::endl;

                if ( NULL != tArray )
                {
                    // only store tiles if length is >= 3
                    ScoredTile sTile( FieldPos(i-length+1, pos), length,
                                      ScoredTile::FIELDDIRECTION_RIGHT,
                                      m_field[i][pos], bombValue );
                    tArray->push_back(sTile);
                }
                else
                {
                    // std::cout << "early break" << std::endl;

                    // we are not interested in any scores
                    // and only want to know if a swapping
                    // is possible
                    break;
                }
            }                
        }
        else
        {
            // new tile here
            if ( length >= 3 )
            {
                found = true;

                // std::cout << "GameField::findSameTilesInRows()    "
                //           << i-length << " "  
                //           << pos << " " 
                //           << length << " " 
                //           << (char)m_field[i-1][j].getType() << " "
                //           << bombValue
                //           << std::endl;

                if ( NULL != tArray )
                {
                    // only store tiles if length is >= 3
                    ScoredTile sTile( FieldPos(i-length, pos), length,
                                      ScoredTile::FIELDDIRECTION_RIGHT,
                                      m_field[i-1][pos], bombValue );
                    tArray->push_back(sTile);
                }
                else
                {
                    // std::cout << "early break" << std::endl;

                    // we are not interested in any scores
                    // and only want to know if a swapping
                    // is possible
                    break;
                }
            }
            
            // set new length
            length = 1;
            bombValue = 0;

            // check if bomb and add bomb value
            if ( m_field[i][pos].isBomb() )
            {
                bombValue += m_field[i][pos].getBombValue();
            }

        }
    } 
    
    return found;
}

// search for equal tiles in single row
// return true if something has been found
const bool GameField::findSameTilesInColumn( const unsigned int pos, ScoredTileArray* tArray ) const
{
    // because this is a internal function we suppose, that pos is
    // between 0 and FIELD_HEIGHT

    bool found = false;
            
    // reset length
    unsigned int length = 0;
    unsigned int bombValue = 0;
    
    for ( unsigned int j = 0; j < FIELD_HEIGHT; j++ )
    {
        if ( ( !m_field[pos][j].isEmpty() ) &&
             ( 0 == j || m_field[pos][j] == m_field[pos][j-1] ) )
        {
            // same tile as before or new start tile
            length++;
            
            // check if bomb and add bomb value
            if ( m_field[pos][j].isBomb() )
            {
                bombValue += m_field[pos][j].getBombValue();
            }
            
            // if the last tile in this column
            // we must check the length
            if ( FIELD_HEIGHT-1 == j && length >= 3 )
            {
                found = true;
                
                // std::cout << "GameField::findSameTilesInColumns() " 
                //           << pos << " "  
                //           << j-length+1 << " " 
                //           << length << " " 
                //           << (char)m_field[i][j].getType() << " "
                //           << bombValue
                //           << std::endl;

                if ( NULL != tArray )
                {
                    // only store tiles if length is >= 3
                    ScoredTile sTile( FieldPos(pos, j-length+1), length,
                                      ScoredTile::FIELDDIRECTION_UP,
                                      m_field[pos][j], bombValue );
                    tArray->push_back(sTile);
                }
                else
                {
                    // std::cout << "early break" << std::endl;

                    // we are not interested in any scores
                    // and only want to know if a swapping
                    // is possible
                    break;
                }
                     
            }                
        }
        else
        {
            // new tile here
            if ( length >= 3 )
            {
                found = true;

                // std::cout << "GameField::findSameTilesInColumns() " 
                //           << pos << " "  
                //           << j-length << " " 
                //           << length << " " 
                //           << (char)m_field[i][j-1].getType() << " "
                //           << bombValue
                //           << std::endl;

                if ( NULL != tArray )
                {
                    // only store tiles if length is >= 3
                    ScoredTile sTile( FieldPos(pos, j-length), length,
                                      ScoredTile::FIELDDIRECTION_UP,
                                      m_field[pos][j-1], bombValue );
                    tArray->push_back(sTile);
                }
                else
                {
                    // std::cout << "early break" << std::endl;

                    // we are not interested in any scores
                    // and only want to know if a swapping
                    // is possible
                    break;
                }
            }

            // set new length
            length = 1;
            bombValue = 0;

            // check if bomb and add bomb value
            if ( m_field[pos][j].isBomb() )
            {
                bombValue += m_field[pos][j].getBombValue();
            }
        }
    }
    
    return found;    
}

// let all existing tiles fall to the ground
void GameField::fallTilesToGround(void)
{
    // let all existing tiles fall to the ground
    for ( unsigned int i = 0; i < FIELD_WIDTH; i++ )
    {
        // position in each column of tile to fill
        unsigned int fill = 0;
        
        for ( unsigned int j = 0; j < FIELD_HEIGHT; j++ )
        {
            if ( !m_field[i][j].isEmpty() )
            {
                if ( fill != j )
                {
                    m_field[i][fill] = m_field[i][j];
                }
                fill++;
            }
        }

        // fill rest of column with zeros
        for ( unsigned int j = fill; j < FIELD_HEIGHT; j++ )
        {
            m_field[i][j].setEmpty();
        }
    }
}

// search for equal entries and removes them on the set
// return true if something has been removed
void GameField::removeTiles(const ScoredTileArray& tArray)
{
    // delete the positions stored in the array
    for ( std::vector<ScoredTile>::const_iterator it = tArray.begin(); it < tArray.end(); it++ )
    {
        // std::cout << "GameField::removeTiles() "
        //           << (*it).getPos().x() << " " << (*it).getPos().y() << std::endl;
        removeTiles(*it);
    }
}

// remove all stored tiles
void GameField::removeTiles(const ScoredTile& tile)
{
    // delete in column
    if ( tile.isUp() )
    {
        // std::cout << "GameField::removeTiles() delete col: "
        //           << tile.getPos().x() << " "
        //           << tile.getPos().y() << " till "
        //           << tile.getPos().x() << " "
        //           << tile.getPos().y()+tile.getLength()-1
        //           << std::endl;
        
        for ( unsigned int j = 0; j < tile.getLength(); j++ )
        {
            m_field[tile.getPos().x()][tile.getPos().y()+j].setEmpty();
        }
    }
    else if ( tile.isRight() )
    {
        // std::cout << "GameField::removeTiles() delete row: "
        //           << tile.getPos().x() << " "
        //           << tile.getPos().y() << " till "
        //           << tile.getPos().x()+tile.getLength()-1 << " "
        //           << tile.getPos().y()
        //           << std::endl;

        for ( unsigned int i = 0; i < tile.getLength(); i++ )
        {
            m_field[tile.getPos().x()+i][tile.getPos().y()].setEmpty();
        }
    }
    else
    {
        std::cout << "GameField::removeSameTiles() error: unknown direction" << std::endl;
    }
}

// remove tiles from array and search new same tiles
// return true if new tiles has been found
const bool GameField::removeAndFindSameTiles( ScoredTileArray& array )
{
    // remove tiles
    removeTiles(array);
    
    // let all existing tiles fall to the ground
    fallTilesToGround();

    // clear array again
    array.clear();

    // std::cout << "Game::removeAndFindSameTiles() info: Intermediate game field" << std::endl;
    // tempField.print();

    // check all fields
    return findSameTiles(&array);
}
        
// calculate value for this field for one swapping position
// return true if everything is ok
const bool GameField::findSameTilesCascading( ScoredTileArray& array,
                                              const bool cascading )
{
    bool ok = false;

    // tiles that lead to scoring
    ScoredTileArray tempArray;

    // flag that at least one tile-strip has been found
    // that lead to score
    bool found = false;

    // in the first run we can use the swapping position
    // for checking the elements
    found = findSameTiles(&tempArray);
                
    while ( found )
    {        
        // add temp array to global one
        // TODO: make own method in class
        for ( std::vector<ScoredTile>::const_iterator it = tempArray.begin(); it < tempArray.end(); it++ )
        {
            array.push_back(*it);
        }
        
        if ( cascading )
        {
            // remove and search new tiles for removing
            found = removeAndFindSameTiles(tempArray);
        }
        else
        {
            // no cascading, leave loop
            found = false;
        }
    }
    
    if ( !array.empty() )
    {
        ok = true;
    }

    return ok;
}
                
// calculate value for this field for one swapping position
// start with this swapping position befor cascading
// return true if everything is ok
const bool GameField::findSameTilesCascading( const SwappingPos& pos,
                                              ScoredTileArray& array,
                                              const bool cascading )
{
    bool ok = false;

    // tiles that lead to scoring
    ScoredTileArray tempArray;
            
    // flag that at least one tile-strip has been found
    // that lead to score
    bool found = false;

    // in the first run we can use the swapping position
    // for checking the elements
    found = findSameTiles(pos, &tempArray);
                
    while ( found )
    {        
        // add temp array to global one
        // TODO: make own method in class
        for ( std::vector<ScoredTile>::const_iterator it = tempArray.begin(); it < tempArray.end(); it++ )
        {
            array.push_back(*it);
        }
        
        if ( cascading )
        {
            // remove and search new tiles for removing
            found = removeAndFindSameTiles(tempArray);
        }
        else
        {
            // no cascading, leave loop
            found = false;
        }
    }
    
    if ( !array.empty() )
    {
        ok = true;
    }

    return ok;
}

