// *************************************************************************
// freiesMagazin-Programmierwettbewerb (game)
// Copyright 2009 Dominik Wagenfuehr <dominik.wagenfuehr@deesaster.org>
// Licence: GPLv3
// *************************************************************************

/**
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef GAME_H
#define GAME_H

// System
///////////
#include <string>

// Eigene
///////////
#include "player.h"
#include "gamefield.h"
#include "swappingposarray.h"
#include "scoredtilearray.h"

class Game
{
public:    // enumerators    

    // game strength of KI
    enum GameStrength
    {
        // PHASE0 does not not calculate any value for the positions but
        // uses the first one that is given
        GAME_STRENGTH_PHASE0 = 0,

        // PHASE1 swaps only two tiles and calculates direct points
        GAME_STRENGTH_PHASE1 = 1,

        // PHASE2 is same as PHASE1 but cascades until no more cascading
        // is possible and calculates this points
        GAME_STRENGTH_PHASE2 = 2,

        // PHASE3 is same as PHASE2 but further calculates recursivly
        // some steps into the future to check what's the best choice
        // for the opponent and considers this too
        GAME_STRENGTH_PHASE3 = 3
    };
    
public:     // methods

    // standard constructor
    Game(void);
    
    // destructor
    ~Game(void);

    // load data from disk
    // return false if something went wrong
    const bool readData( const std::string& fieldFilename,
                         const std::string& playerFilename,
                         const std::string& opponentFilename );

    // calculate the best position and stores it on disk
    // return false if something went wrong
    const bool calculateAndSaveBestPosition( const std::string& filename ) const;

    // set game strength
    void setStrength( const GameStrength strength );
    
    // set max depth for PHASE3
    // must be >= 0
    void setMaxDepth( const int depth );

private:    // methods

    // choose the best position and stores it on disk
    // return false if something went wrong
    static const bool chooseAndSaveBestPosition( const SwappingPosArray& positions,
                                                 const std::string& filename );

    // choose the best position from calculated positions
    // return false if something went wrong
    static const bool chooseBestPosition( SwappingPos& pos, const SwappingPosArray& positions );

    // store positions in file
    // return true if everything is ok
    static const bool writePositions(const SwappingPos& pos, const std::string& filename);

    // calculate the possible positions and corresponding values
    // we will give field and players as argument because this routine will
    // work recursivly in PHASE3
    // if not PHASE3 we will work on the reference and do not make a copy!
    // return false if somethign went wrong
    const bool calculatePositionsAndValues( const GameField& field,
                                            const Player& player,
                                            const Player& opponent,
                                            SwappingPosArray& positions,
                                            const int depth = -1 ) const;
                                                   
    // calculate the values for possible positions
    // return false if something went wrong
    const bool calculateSwapPositionValues( const GameField& field,
                                            const Player& player,
                                            const Player& opponent,
                                            SwappingPosArray& positions,
                                            const int depth = -1 ) const; 
                                                                                              
    // convert scored tiles to a number corresponding to player stats
    // this is the weighting function of the whole KI
    // and therefor the most important part!
    static const int convertToScore( const Player& player,
                                     const ScoredTileArray& tArray );                                  
                                                
private:    // members

    // player and opponent
    Player m_player;
    Player m_opponent;
    
    // one game field
    GameField m_field;
    
    // strength of KI
    GameStrength m_strength;
    
    // depth for precalculation
    int m_maxDepth;
};

#endif // GAME_H
