// *************************************************************************
// freiesMagazin-Programmierwettbewerb (game)
// Copyright 2009 Dominik Wagenfuehr <dominik.wagenfuehr@deesaster.org>
// Licence: GPLv3
// *************************************************************************

/**
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// System
///////////
#include <iostream>
#include <fstream>
#include <cstdio>
#include <stdlib.h>
#include <limits.h>

// Eigene
///////////
#include "game.h"

// standard constructor
Game::Game(void)
    : m_strength (GAME_STRENGTH_PHASE0), m_maxDepth(1)
{
}

// destructor
Game::~Game(void)
{
}

// load data from disk
// return false if something went wrong
const bool Game::readData( const std::string& fieldFilename,
                           const std::string& playerFilename,
                           const std::string& opponentFilename )
{
    bool ok = true;

    // load game field
    if ( !m_field.read( fieldFilename ) )
    {
        ok  = false;
    }

    // load player
    if ( !m_player.read("player.dat") )
    {
        ok  = false;
    }

    // load opponent
    if ( !m_opponent.read("opponent.dat") )
    {
        ok  = false;
    }

    return ok;    
}

// set game strength
void Game::setStrength( const GameStrength strength )
{
    m_strength = strength;
}

// set max depth for PHASE3
// must be >= 0
void Game::setMaxDepth( const int depth )
{
    // this value says that we will calculate
    // X steps into the future
    // 0 means own choice
    // 1 means own choice and next choice of opponent
    // 2 means own choice and next choice of opponent and next own choice
    // and so on
    // must at least be 0
    // value 0 is equal to PHASE2 !
    
    if ( depth >= 0 )
    {
        m_maxDepth = depth;
    }
    else
    {
        std::cout << "Game::setMaxDepth() error: max depth is negative"
                  << std::endl;
    }
}

// calculate the best position and stores it on disk
// return false if something went wrong
const bool Game::calculateAndSaveBestPosition( const std::string& filename ) const
{
    bool ok = false;

    // possible positions and values
    SwappingPosArray positions;
       
    // calculate the possible positions and corresponding values
    // we will give field and players as argument because this routine will
    // work recursivly in PHASE3
    // if not PHASE3 we will work on the reference and do not make a copy!
    if ( calculatePositionsAndValues( m_field, m_player, m_opponent, positions ) )
    {
        // choose best position ( = highest value ) from precalculated
        // positions and values
        if ( chooseAndSaveBestPosition( positions, filename ) )
        {
            ok = true;
        }
    }
    
    return ok;    
}

// store positions in file
// return true if everything is ok
const bool Game::writePositions(const SwappingPos& pos, const std::string& filename)
{
    bool ok = false;

    if ( !filename.empty() )
    {
        // open file for writing
        std::ofstream outfile( filename.c_str() );
        
        if ( outfile.good() )
        {
            // write positions into file
            outfile << pos.first().x() << " "
                    << pos.first().y() << " "
                    << pos.second().x() << " "
                    << pos.second().y() << std::endl;
            
            // close file
            outfile.close();
            
            ok = true;
        }
        else
        {
            std::cout << "Game::writePositions() error: file "
                      << filename << " could not be opened" << std::endl;
        }
    }
    else
    {
        std::cout << "Game::writePositions() error: string is empty" << std::endl;
    }
    
    return ok;
}

// choose the best position and stores it on disk
// return false if something went wrong
const bool Game::chooseAndSaveBestPosition( const SwappingPosArray& positions,
                                            const std::string& filename )
{
    bool ok = false;

    // choose best position ( = highest value ) from precalculated
    // positions and values
    SwappingPos pos;
    if ( chooseBestPosition( pos, positions ) )
    {
        // finally write positions to disk
        if ( writePositions( pos, filename ) )
        {
            ok = true;
        }
    }
    
    return ok;
}

// choose the best position from calculated positions
// return false if something went wrong
const bool Game::chooseBestPosition( SwappingPos& pos, const SwappingPosArray& positions )
{
    bool ok = false;

    if ( positions.empty() )
    {
        std::cout << "Game::chooseBestPosition() error: Array is empty" << std::endl;
    }
    else
    {
        // flag that this is the first value
        bool first = true;
        
        // iterate over all positions
        for ( std::vector<SwappingPos>::const_iterator it = positions.begin(); it < positions.end(); it++ )
        {
            // check if there is better value
            if ( first || ( pos.getValue()  < (*it).getValue() ) )
            {
                // store position
                pos = *it;
                ok = true;
            }
            first = false;
        }
    }

    return ok;
}

// calculate the possible positions and corresponding values
// we will give field and players as argument because this routine will
// work recursivly in PHASE3
// if not PHASE3 we will work on the reference and do not make a copy!
// return false if something went wrong
const bool Game::calculatePositionsAndValues( const GameField& field,
                                              const Player& player,
                                              const Player& opponent,
                                              SwappingPosArray& positions,
                                              int depth ) const
{
    bool ok = false;

    // find swapping positions
    if ( field.findSwapPositions(positions) )
    {
        // now we will check for each swapped position what the 
        // result would be if we swap it
        if ( calculateSwapPositionValues(field, player, opponent, positions, depth) )
        {
            ok = true;
        }
    }
    
    return ok;
}

// calculate the values for possible positions
// return false if something went wrong
const bool Game::calculateSwapPositionValues( const GameField& field,
                                              const Player& player,
                                              const Player& opponent,
                                              SwappingPosArray& positions,
                                              int depth ) const
{
    bool ok = true;

    if ( GAME_STRENGTH_PHASE3 == m_strength )
    {
        depth++;
            
        // we have precalculated enough
        if ( depth > m_maxDepth )
        {
            return false;
        }

        // std::cout << "Dive into depth " << depth << std::endl;
    }
        
    // iterate over all positions and swap them
    for ( std::vector<SwappingPos>::iterator it = positions.begin(); it < positions.end(); it++ )
    {
        // copy game field
        GameField tempField(field);

        if ( tempField.swapTiles(*it) )
        {
            // calculate all tiles that would lead to scores (as cascade is necessary)
            ScoredTileArray tArray;

            // convert this scored tiles to a number
            int value = 0;

            // cascade tiles until nothing can be removed anymore
            bool cascade = false;
            
            if ( GAME_STRENGTH_PHASE2 == m_strength ||
                 GAME_STRENGTH_PHASE3 == m_strength )
            {
                cascade = true;
            }         

            if ( tempField.findSameTilesCascading( *it, tArray, cascade ) )
            {
                if ( GAME_STRENGTH_PHASE0 != m_strength )
                {
                    value += convertToScore(player, tArray);
                }
            }

            if ( GAME_STRENGTH_PHASE3 == m_strength )
            {
                //std::cout << "Start Value: " << value
                //          << " at depth " << depth
                //          << std::endl;

                // copy player and opponent
                Player tempPlayer(player);
                Player tempOpponent(opponent);            

                // add points for player
                tempPlayer.addPoints(tArray);
                
                // calculate damage for opponent
                tempOpponent.looseLife( tempPlayer.getOpponentDamage() );            

                // maybe someone is dead already so we do not need to
                // work recursivly
                if ( !tempOpponent.isDead() && !tempPlayer.isDead() )
                {
                    // std::cout << "Game::cascade() info: recursive game field" << std::endl;
                    // tempField.print();

                    // possible positions and values
                    SwappingPosArray tempPositions;
                       
                    // calculate the possible positions and corresponding values for opponent
                    if ( calculatePositionsAndValues( tempField, tempOpponent, tempPlayer, tempPositions, depth ) )
                    {
                        // choose best position ( = highest value ) from precalculated
                        // positions and values
                        SwappingPos tempPos;
                        if ( chooseBestPosition( tempPos, tempPositions ) )
                        {
                            // finally subtract this value from the previous calculated result
                            value -= tempPos.getValue();
                        }
                    }
                }
            } // PHASE3
            
            // add value too result
            (*it).addValue( value );

#ifdef DEBUG            
            std::cout << "End Value: " << value
                      << " at depth " << depth
                      << std::endl;
#endif
            
        }
        else
        {
            std::cout << "Error: tiles are not swapable at "
                      << (*it).first().x() << " " << (*it).first().y() << " and "
                      << (*it).second().x() << " " << (*it).second().y()
                      << std::endl;
                      
            // tiles not swapable => leave loop
            ok = false;
            break;
        }
        
    } // for

    if ( GAME_STRENGTH_PHASE3 == m_strength )
    {
        depth--;
    }

    return ok;
}

// convert scored tiles to a number corresponding to player stats
// this is the weighting function of the whole KI
// and therefor the most important part!
const int Game::convertToScore( const Player& player,
                          const ScoredTileArray& tArray )
{                
    int value = 0;

    // iterate over all scored tiles
    for ( std::vector<ScoredTile>::const_iterator it = tArray.begin(); it < tArray.end(); it++ )
    {
        // std::cout << "Tile: " << (char)(*it).getTile().getType()
        //           << " Len: " << (*it).getLength()
        //           << " Bomb: " << (*it).getBombValue()
        //           << std::endl;

        if ( (*it).getTile().isBomb() )
        {
            // it's a bomb
            value += (*it).getBombValue() * 10;
        }
        else
        {
            // it's another tile
            switch ( (*it).getTile().getType() )
            {
                case Tile::TILE_RED:
                    value += (*it).getLength() * 3;    
                    if ( player.getRed() + (*it).getLength() >= Player::PLAYER_MAX_POINTS )
                    {
                        value += 10;
                    }
                    break;
                case Tile::TILE_YELLOW:
                    value += (*it).getLength() * 2;
                    if ( player.getYellow() + (*it).getLength() >= Player::PLAYER_MAX_POINTS )
                    {
                        value += 6;
                    }
                    break;
                case Tile::TILE_GREEN:
                    value += (*it).getLength() * 1;
                    if ( player.getGreen() + (*it).getLength() >= Player::PLAYER_MAX_POINTS )
                    {
                        value += 3;
                    }
                    break;
                case Tile::TILE_SHIELD:
                    value += ( Player::PLAYER_MAX_SHIELD - player.getShield() ) / 3 * (*it).getLength();
                    break;
                case Tile::TILE_LILAC:
                    value += 1;
                    if ( player.getLilac() + (*it).getLength() >= Player::PLAYER_EXTRA_TURN )
                    {
                        value += 14;
                    }
                    break;
                default:
                    break;
            } // switch
        } // if
        
    } // for
    
    return value;
}

